package jadean.dean.java.resourceparser.classparser;


import jadean.dean.Resource;
import jadean.dean.exceptions.ResourceIOException;
import jadean.dean.exceptions.ResourceInvalidException;
import jadean.dean.exceptions.ResourceNotFoundException;
import jadean.dean.java.resourceparser.JavaResourceFilter;
import jadean.dean.java.resourceparser.JavaResourceParser;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfo;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoClass;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoDouble;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoFieldRef;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoFloat;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoInteger;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoInterfaceMethodRef;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoLong;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoMethodRef;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoNameAndType;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoString;
import jadean.dean.java.resourceparser.classparser.cpinfo.CPInfoUTF8;
import jadean.dean.java.resourceparser.classparser.cpinfo.constants.CPInfoConstants;
import jadean.dean.java.resources.JavaResourceContext;
import jadean.dean.java.resources.JavaResourceDefault;
import jadean.dean.java.utilities.DebugUtilities;

import java.io.DataInput;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashSet;


public class ClassResourceParser extends JavaResourceParser {

	private DataInput di;
	private InputStream is;
	private String res;
	private int magic;
	private int minor;
	private int major;
	private int constantPoolCount;
	private int accessFlags;
	private int thisClass;
	private int superClass;
	private int interfacesCount;
	private int fieldsCount;
	private int methodsCount;
	private int attributesCount;
	private CPInfo[] constantPoolTable;
	
	public ClassResourceParser(String res) throws ResourceNotFoundException {
		super(res);
		try {
			this.res = res;
			this.is = new FileInputStream(new File(res));
			this.di = new DataInputStream(this.is);
		}
		catch (IOException e) {
			throw new ResourceNotFoundException();
		}
	}
	
	public ClassResourceParser(String res, InputStream is) throws ResourceNotFoundException {
		super(res);
		if (is == null) throw new ResourceNotFoundException();
		this.res = res;
		this.is = is;
		this.di = new DataInputStream(this.is);	
	}
	
	@SuppressWarnings("unused")
	private void debug() {
		System.out.println("magic = " + magic);
		System.out.println("minor = " + minor);
		System.out.println("major = " + major);
		System.out.println("constantPoolCount = " + constantPoolCount);
		System.out.println("accessFlags = " + accessFlags);
		System.out.println("thisClass = " + thisClass);
		System.out.println("superClass = " + superClass);
		System.out.println("interfacesCount = " + interfacesCount);
		System.out.println("fieldsCount = " + fieldsCount);
		System.out.println("methodsCount = " + methodsCount);
		System.out.println("attributesCount = " + attributesCount);
		DebugUtilities.print(constantPoolTable);
	}		

	private void parse() throws ResourceIOException, ResourceInvalidException {
		try {
			this.magic = this.di.readInt();
			if (this.magic != CPInfoConstants.MAGIC) throw new ResourceInvalidException("MAGIC");
			this.minor = this.di.readUnsignedShort();
			this.major = this.di.readUnsignedShort();
			this.constantPoolCount = this.di.readUnsignedShort();
			this.constantPoolTable = new CPInfo[this.constantPoolCount - 1];
			for (int i = 0; i < this.constantPoolCount - 1; i++) {
				CPInfo ci = this.readCPInfo();
				this.constantPoolTable[i] = ci;
				if (ci instanceof CPInfoLong || ci instanceof CPInfoDouble) i++;
			}
			this.accessFlags = this.di.readUnsignedShort();
			this.thisClass = this.di.readUnsignedShort();
			this.superClass = this.di.readUnsignedShort();
			this.interfacesCount = this.di.readUnsignedShort();
			this.readInterfaces();
			this.fieldsCount = this.di.readUnsignedShort();
			this.readFields();
			this.methodsCount = this.di.readUnsignedShort();
			this.readMethods();
			this.attributesCount = this.di.readUnsignedShort();
			this.readAttributes(this.attributesCount);	
			this.is.close();
		}
		catch (EOFException ee) {
			throw new ResourceInvalidException("Invalid format");
		}
		catch (IOException ie) {
			throw new ResourceIOException();
		}
	}
	
	private CPInfo readCPInfo() throws EOFException, IOException {
		int tag = this.di.readUnsignedByte();
		switch(tag) {
		case CPInfoConstants.CPINFO_UTF8:
			return new CPInfoUTF8(this.di);
		case CPInfoConstants.CPINFO_INTEGER:
			return new CPInfoInteger(this.di);
		case CPInfoConstants.CPINFO_FLOAT:
			return new CPInfoFloat(this.di);
		case CPInfoConstants.CPINFO_LONG:
			return new CPInfoLong(this.di);
		case CPInfoConstants.CPINFO_DOUBLE:
			return new CPInfoDouble(this.di);
		case CPInfoConstants.CPINFO_CLASS:
			return new CPInfoClass(this.di);
		case CPInfoConstants.CPINFO_STRING:
			return new CPInfoString(this.di);
		case CPInfoConstants.CPINFO_FIELD_REF:
			return new CPInfoFieldRef(this.di);
		case CPInfoConstants.CPINFO_METHOD_REF:
			return new CPInfoMethodRef(this.di);
		case CPInfoConstants.CPINFO_INTERFACE_METHOD_REF:
			return new CPInfoInterfaceMethodRef(this.di);
		case CPInfoConstants.CPINFO_NAME_AND_TYPE:
			return new CPInfoNameAndType(this.di);
		}
		return null;
	}
	
	private void readInterfaces() throws EOFException, IOException {
		for (int i = 0; i < interfacesCount; i++) {
			this.di.readUnsignedShort();
		}
	}
	
	private void readFields() throws EOFException, IOException {
		for (int i = 0; i < fieldsCount; i++) {
			this.di.readUnsignedShort();
			this.di.readUnsignedShort();
			this.di.readUnsignedShort();
			int attributesCount = this.di.readUnsignedShort();
			this.readAttributes(attributesCount);
		}		
	}
	
	private void readMethods() throws EOFException, IOException {
		for (int i = 0; i < methodsCount; i++) {
			this.di.readUnsignedShort();
			this.di.readUnsignedShort();
			this.di.readUnsignedShort();
			int attributesCount = this.di.readUnsignedShort();
			this.readAttributes(attributesCount);
		}			
	}
	
	private void readAttributes(int j) throws EOFException, IOException {
		for (int i = 0; i < j; i++) {
			this.di.readUnsignedShort();
			int k = this.di.readInt();
			for (int l = 0; l < k; l++) {
				this.di.readUnsignedByte();
			}
		}			
	}	

	@Override
	public Collection<Resource> getReferencedResources() throws ResourceIOException, ResourceInvalidException {
		this.parse();
		HashSet<Resource> a = new HashSet<Resource>();
		int ib = this.res.lastIndexOf(File.separatorChar);
		if (ib < 0) ib = 0;
		JavaResourceFilter rf = new ClassResourceFilter(this.res.substring(ib));
		for (int i = 0; i < this.constantPoolTable.length; i++) {
			if (this.constantPoolTable[i] instanceof CPInfoClass) {
				CPInfoClass cpic = (CPInfoClass) this.constantPoolTable[i];
				CPInfoUTF8 cpiu = (CPInfoUTF8) this.constantPoolTable[cpic.getIndex() - 1];
				String s = ClassResourceUtilities.formatClass(cpiu.getData());
				JavaResourceContext cnc = new JavaResourceContext();
				if (rf.accept(new JavaResourceDefault(s, cnc))) {
					a.add(new JavaResourceDefault(s, cnc));
				}
			}
		}	
		return a;
	}
	
	private boolean isReflectionCall(String className, String methodName) {
		return (className.equals("java/lang/Class") && methodName.equals("forName")) ||
				(className.equals("java/lang/Class") && methodName.equals("getSuperClass")) ||
				(className.equals("java/lang/Class") && methodName.equals("getClasses")) ||
				(className.equals("java/lang/Class") && methodName.equals("getDeclaredClasses")) ||
				(className.equals("java/lang/Class") && methodName.equals("getEnclosingClass")) ||
				(className.equals("java/lang/Class") && methodName.equals("getDeclaringClasses")) ||
				(className.equals("java/lang/reflect/Field") && methodName.equals("getDeclaringClass")) ||
				(className.equals("java/lang/reflect/Method") && methodName.equals("getDeclaringClass")) ||
				(className.equals("java/lang/reflect/Constructor") && methodName.equals("getDeclaringClass"));
	}
	
	public boolean isReflectionUsed() {
		for (CPInfo cpInfo: constantPoolTable) {
			if (cpInfo instanceof CPInfoMethodRef) {
				CPInfoMethodRef methodRef = (CPInfoMethodRef) cpInfo;
				CPInfoClass infoClass = (CPInfoClass) this.constantPoolTable[methodRef.getClassIndex() - 1];
				CPInfoNameAndType infoNameAndType = (CPInfoNameAndType) this.constantPoolTable[methodRef.getNameAndTypeIndex() - 1];
				CPInfoUTF8 classNameUTF8 = (CPInfoUTF8) this.constantPoolTable[infoClass.getIndex() - 1];
				CPInfoUTF8 methodNameUTF8 = (CPInfoUTF8) this.constantPoolTable[infoNameAndType.getNameIndex() - 1];
				if (isReflectionCall(classNameUTF8.getData(), methodNameUTF8.getData())) {
					return true;
				}
			}
		}
		return false;
	}

}
